// server.js - Magnata IA (exemplo mínimo)
require('dotenv').config();
const express = require('express');
const axios = require('axios');
const { v4: uuidv4 } = require('uuid');
const rateLimit = require('express-rate-limit');
const cors = require('cors');

const app = express();
app.use(cors());
app.use(express.json());
app.use(express.static('public'));

const PORT = process.env.PORT || 3000;
const OPENAI_API_KEY = process.env.OPENAI_API_KEY || '';
const MODEL = process.env.MODEL || 'gpt-4o-mini';

const limiter = rateLimit({ windowMs: 1000*60, max: 60 });
app.use(limiter);

const sessions = new Map();

const SYSTEM_PROMPT = `Você é Magnata IA: assistente profissional, direto e preciso.
Responda em Português do Brasil. Seja útil, objetivo e explique passos quando pedir código.`;

async function callLLM(messages) {
  if (!OPENAI_API_KEY) throw new Error('OPENAI_API_KEY não definido');

  const payload = { model: MODEL, messages, max_tokens: 1000, temperature: 0.2 };

  const res = await axios.post('https://api.openai.com/v1/chat/completions', payload, {
    headers: { Authorization: `Bearer ${OPENAI_API_KEY}`, 'Content-Type': 'application/json' }
  });

  return res.data.choices[0].message.content;
}

app.post('/session', (req, res) => {
  const sessionId = uuidv4();
  sessions.set(sessionId, { history: [{ role: 'system', content: SYSTEM_PROMPT }] });
  res.json({ sessionId });
});

app.post('/chat', async (req, res) => {
  try {
    const { sessionId, message } = req.body;
    if (!sessionId || !message) return res.status(400).json({ error: 'sessionId e message são necessários' });
    const session = sessions.get(sessionId);
    if (!session) return res.status(404).json({ error: 'Sessão não encontrada' });

    const forbidden = ['explosive','hack','bomb'];
    const lc = message.toLowerCase();
    if (forbidden.some(w => lc.includes(w))) {
      return res.json({ reply: 'Não posso ajudar com essa solicitação. Posso oferecer alternativas legais e seguras.' });
    }

    session.history.push({ role: 'user', content: message });

    const MAX = 12;
    const recent = session.history.slice(-MAX-1);
    const llmMessages = [{ role:'system', content:SYSTEM_PROMPT }].concat(recent.filter(m=>m.role!=='system'));

    const reply = await callLLM(llmMessages);

    session.history.push({ role: 'assistant', content: reply });

    res.json({ reply });

  } catch (err) {
    console.error(err);
    res.status(500).json({ error:'Erro interno', detail:err.message });
  }
});

app.listen(PORT, () => console.log(`Magnata IA rodando em http://localhost:${PORT}`));
